
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include <jpeglib.h>
#include "jpeginput.h"

struct jpeginput_s
{
    int width;
    int height;
    uint8_t *buffer;
};

struct my_error_mgr {
    struct jpeg_error_mgr pub;	/* "public" fields */
    jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

void my_error_exit( j_common_ptr cinfo )
{
  my_error_ptr myerr = (my_error_ptr) cinfo->err;
  (*cinfo->err->output_message)( cinfo );
  longjmp( myerr->setjmp_buffer, 1 );
}

jpeginput_t *jpeginput_new( const char *filename, int ycbcr )
{
    jpeginput_t *jpeginput = malloc( sizeof( jpeginput_t ) );
    struct jpeg_decompress_struct cinfo;
    struct my_error_mgr jerr;
    JSAMPARRAY buffer;
    int row_stride;
    FILE *infile;

    if( !jpeginput ) {
        return 0;
    }

    infile = fopen( filename, "rb" );
    if( !infile ) {
        fprintf( stderr, "jpeginput: Can't open %s\n", filename );
        free( jpeginput );
        return 0;
    }

    cinfo.err = jpeg_std_error( &jerr.pub );
    jerr.pub.error_exit = my_error_exit;
    if( setjmp( jerr.setjmp_buffer ) ) {
        jpeg_destroy_decompress( &cinfo );
        fclose( infile );
        return 0;
    }

    jpeg_create_decompress( &cinfo );
    jpeg_stdio_src( &cinfo, infile );
    jpeg_read_header( &cinfo, TRUE );

    fprintf( stderr, "input: %d\n", cinfo.jpeg_color_space );

    if( ycbcr ) {
        cinfo.out_color_space = JCS_YCbCr;
    }

    fprintf( stderr, "jpeginput: colour space %d (YCbCr %d)\n", cinfo.jpeg_color_space, JCS_YCbCr );
    fprintf( stderr, "jpeginput: 601 sampling %d\n", cinfo.CCIR601_sampling );
    fprintf( stderr, "jpeginput: output colour space %d (RGB %d)\n", cinfo.out_color_space, JCS_RGB );

    jpeg_start_decompress( &cinfo );

    row_stride = cinfo.output_width * cinfo.output_components;
    buffer = (*cinfo.mem->alloc_sarray)( (j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1 );

    jpeginput->width = cinfo.image_width;
    jpeginput->height = cinfo.image_height;
    jpeginput->buffer = malloc( jpeginput->width * jpeginput->height * 3 );

    while( cinfo.output_scanline < cinfo.output_height ) {
        jpeg_read_scanlines( &cinfo, buffer, 1 );
        memcpy( jpeginput->buffer + ((cinfo.output_scanline - 1) * jpeginput->width * 3), buffer[ 0 ], row_stride );
    }
    jpeg_finish_decompress( &cinfo );
    jpeg_destroy_decompress( &cinfo );
    fclose( infile );
    return jpeginput;
}

void jpeginput_delete( jpeginput_t *jpeginput )
{
    free( jpeginput->buffer );
    free( jpeginput );
}

unsigned int jpeginput_get_width( jpeginput_t *jpeginput )
{
    return jpeginput->width;
}

unsigned int jpeginput_get_height( jpeginput_t *jpeginput )
{
    return jpeginput->height;
}

uint8_t *jpeginput_get_scanline( jpeginput_t *jpeginput, int num )
{
    return jpeginput->buffer + (jpeginput->width * num * 3);
}

